#include <X11/Xlib.h>
#include <X11/keysym.h>
#include <X11/Xft/Xft.h>
#include <security/pam_appl.h>
#include <stdbool.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <pwd.h>

int pam_conv_func(
    int num_msg,
    const struct pam_message **msg,
    struct pam_response **resp,
    void *appdata_ptr)
{
    struct pam_response *responses =
        calloc(num_msg, sizeof(struct pam_response));
    if (!responses)
        return PAM_BUF_ERR;

    for (int i = 0; i < num_msg; i++)
    {
        switch (msg[i]->msg_style)
        {
        case PAM_PROMPT_ECHO_OFF:
            responses[i].resp = strdup((char *)appdata_ptr);
            responses[i].resp_retcode = 0;
            break;

        case PAM_TEXT_INFO:
        case PAM_ERROR_MSG:
            responses[i].resp = NULL;
            responses[i].resp_retcode = 0;
            break;

        default:
            free(responses);
            return PAM_CONV_ERR;
        }
    }

    *resp = responses;
    return PAM_SUCCESS;
}

bool password_prompt(Display *d, int s, Window parent,
                     XftFont *font, XftColor *color)
{
    enum
    {
        PW_INPUT,
        PW_AUTH
    } state = PW_INPUT;

    char input[64] = {0};
    int len = 0;

    /* ---- Create window ---- */
    Window pw = XCreateSimpleWindow(
        d, parent,
        80, 60, 300, 120, 1,
        WhitePixel(d, s),
        BlackPixel(d, s));

    XSelectInput(d, pw, ExposureMask | KeyPressMask);
    XMapRaised(d, pw);

    /* ---- Modal grabs (kiosk critical) ---- */
    XGrabKeyboard(d, pw, True,
                  GrabModeAsync, GrabModeAsync, CurrentTime);
    XGrabPointer(d, pw, True,
                 ButtonPressMask | ButtonReleaseMask,
                 GrabModeAsync, GrabModeAsync,
                 None, None, CurrentTime);

    GC gc = XCreateGC(d, pw, 0, NULL);
    XSetForeground(d, gc, WhitePixel(d, s));

    XftDraw *xft = XftDrawCreate(
        d, pw,
        DefaultVisual(d, s),
        DefaultColormap(d, s));

    XEvent e;

    /* ---- Input / UI loop ---- */
    while (1)
    {
        XNextEvent(d, &e);

        if (e.type != Expose && e.type != KeyPress)
            continue;

        if (state == PW_INPUT && e.type == KeyPress)
        {
            KeySym ks;
            char buf[8];

            int n = XLookupString(&e.xkey, buf, sizeof(buf), &ks, NULL);

            if (ks == XK_Return)
            {
                state = PW_AUTH;
            }
            else if (ks == XK_BackSpace && len > 0)
            {
                input[--len] = 0;
            }
            else if (n == 1 && buf[0] >= 32 && len < (int)sizeof(input) - 1)
            {
                input[len++] = buf[0];
            }
        }

        /* ---- Redraw ---- */
        XClearWindow(d, pw);

        if (state == PW_INPUT)
        {
            XDrawString(d, pw, gc, 20, 25,
                        "Enter Password:", 15);

            char stars[64];
            memset(stars, '*', len);
            stars[len] = '\0';

            XftDrawStringUtf8(
                xft, color, font,
                20, 70,
                (XftChar8 *)stars,
                strlen(stars));
        }
        else
        { /* PW_AUTH */
            XDrawString(d, pw, gc, 20, 55,
                        "Authenticating...", 17);
            XSync(d, False); /* force draw before PAM */
            break;
        }
    }

    /* ---- PAM authentication ---- */
    pam_handle_t *pamh = NULL;
    struct pam_conv conv = {
        .conv = pam_conv_func,
        .appdata_ptr = input};

    int ret = PAM_AUTH_ERR;

    struct passwd *pwd = getpwuid(getuid());
    if (!pwd)
    {
        ret = PAM_AUTH_ERR;
    }
    else
    {
        if (pam_start("kiosk-auth", pwd->pw_name, &conv, &pamh) == PAM_SUCCESS)
        {
            ret = pam_authenticate(pamh, 0);
            pam_end(pamh, ret);
        }
    }

    /* ---- Zero password memory ---- */
    explicit_bzero(input, sizeof(input));

    /* ---- Teardown ---- */
    XUngrabKeyboard(d, CurrentTime);
    XUngrabPointer(d, CurrentTime);

    XSync(d, False);
    XftDrawDestroy(xft);
    XFreeGC(d, gc);
    XDestroyWindow(d, pw);

    /* Drain remaining events related to pw */
    XEvent ev;
    while (XPending(d))
    {
        XNextEvent(d, &ev);
    }

    return ret == PAM_SUCCESS;
}